export default {
  id: 'playing',
  handler: (router, context) => {
    const { services, database, getSchema, env, logger, emitter } = context
    const { ItemsService, UsersService, AuthenticationService } = services

    router.post('/join-team', async (req, res, next) => {
      //
      //A user submits a team id,
      //we validate the team exists and is for an upcomming (game start time is in the future)
      //no limit on how many teams a player can join.
      //if the user is in an existing team for the same game then he should withdraw from the existing team first.
      const adventuresService = new ItemsService('adventures', {
        schema: req.schema,
        accountability: req.accountability
      })

      const teamsService = new ItemsService('teams', {
        schema: req.schema,
        accountability: req.accountability
      })

      const playersService = new ItemsService('players', {
        schema: req.schema,
        accountability: req.accountability
      })

      try {
        const { teamId } = req.body

        const userId = req.accountability.user
        const team = await teamsService.readOne(teamId)

        const adventure = await adventuresService.readOne(team.adventure)

        if (new Date(adventure.start_date_and_time) < new Date()) {
          return res
            .status(400)
            .json({ success: false, error: 'This game has already started.' })
        }

        let existingPlayer = await playersService.readByQuery({
          filter: { user: { _eq: userId }, team: { _eq: teamId } },
          limit: 1
        })

        if (existingPlayer.length > 0) {
          return res
            .status(400)
            .json({
              success: false,
              error: 'You seem to have already joined this team!'
            })
        }

        const myTeams = await playersService.readByQuery({
          fields: ['*', 'team.adventure'],
          filter: {
            user: {
              _eq: userId
            }
          },
          limit: -1
        })

        const adventuresArray = myTeams.map(object => object.team.adventure)

        if (adventuresArray.includes(adventure.id)) {
          return res
            .status(400)
            .json({
              success: false,
              error:
                'You seem to have already joined another team! please withdraw yourself from your current team first'
            })
        }

        const newPlayer = await playersService.createOne({
          user: userId,
          team: teamId
        })
        res.status(200).json({
          success: true,
          player: newPlayer
        })
      } catch (error) {
        res.status(500).json({
          success: false,
          error: error.message
        })
      }
    })

    router.post('/withdraw-team', async (req, res, next) => {
      //A user submits a team id,
      //we validate the team exists and is for an upcomming (game start time is in the future)
      //the related players record is deleted
      const adventuresService = new ItemsService('adventures', {
        schema: req.schema,
        accountability: req.accountability
      })

      const teamsService = new ItemsService('teams', {
        schema: req.schema,
        accountability: req.accountability
      })

      const playersService = new ItemsService('players', {
        schema: req.schema,
        accountability: req.accountability
      })

      try {
        const { teamId } = req.body
        const userId = req.accountability.user

        if (!teamId) {
          return res
            .status(400)
            .json({ success: false, error: 'Team code is required.' })
        }
        const team = await teamsService.readOne(teamId)
        const adventure = await adventuresService.readOne(team.adventure)

        if (new Date(adventure.start_date_and_time) < new Date()) {
          return res
            .status(400)
            .json({
              success: false,
              error:
                'This game has already started. You can not change the team anymore!'
            })
        }

        let existingPlayer = await playersService.readByQuery({
          filter: { user: { _eq: userId }, team: { _eq: teamId } },
          limit: 1
        })

        if (existingPlayer.length == 0) {
          return res
            .status(400)
            .json({
              success: false,
              error: 'Player is not a member of the specified team!'
            })
        }
        await playersService.deleteOne(existingPlayer[0].id)
        res.status(200).json({
          success: true
        })
      } catch (error) {
        res.status(500).json({
          success: false,
          error: error.message
        })
      }
    })

    router.post('/start-game', async (req, res, next) => {
      // A user submits a team id,
      // we validate the team exists and is for an ongoing (current time is between game start and end times)
      // we validate a player exists for the user and team id
      // we set all other user memberships/players is-current flag to false and set found player is-current to true
      // This is needed to help the back end identify the player current game when the player is registered in more than one team
      const adventuresService = new ItemsService('adventures', {
        schema: req.schema,
        accountability: req.accountability
      })

      const teamsService = new ItemsService('teams', {
        schema: req.schema,
        accountability: req.accountability
      })

      const playersService = new ItemsService('players', {
        schema: req.schema,
        accountability: req.accountability
      })

      try {
        const { teamId } = req.body
        const userId = req.accountability.user

        if (!teamId) {
          return res
            .status(400)
            .json({ success: false, error: 'Team code is required.' })
        }
        const team = await teamsService.readOne(teamId)

        const adventure = await adventuresService.readOne(team.adventure)

        if (new Date(adventure.start_date_and_time) > new Date()) {
          return res
            .status(400)
            .json({ success: false, error: 'The game did not start yet!' })
        }
        if (new Date(adventure.finish_date_and_time) < new Date()) {
          return res
            .status(400)
            .json({
              success: false,
              error: 'The game is not available anymore!'
            })
        }

        let existingPlayer = await playersService.readByQuery({
          filter: { user: { _eq: userId }, team: { _eq: teamId } },
          limit: 1
        })

        if (existingPlayer.length == 0) {
          return res
            .status(400)
            .json({
              success: false,
              error: 'The player is not a member of the specified team!'
            })
        }

        for (let i = 0; i < existingPlayer.length; i++) {
          let id = existingPlayer[i].id
          playersService.updateOne(id, { is_current: true })
        }

        // update other user games
        let otherGames = await playersService.readByQuery({
          filter: { user: { _eq: userId }, team: { _neq: teamId } },
          limit: -1
        })

        for (let i = 0; i < otherGames.length; i++) {
          let id = otherGames[i].id
          playersService.updateOne(id, { is_current: false })
        }

        res.status(200).json({
          success: true,
          adventure: adventure,
          player: existingPlayer[0]
        })
      } catch (error) {
        res.status(500).json({ success: false, error: error.message })
      }
    })

    router.post('/load-game', async (req, res, next) => {
      /*
            A user submits a team id, 
            we validate the team exists and is for an ongoing (current time is between game start and end times)
            we validate a player exists for the user and team id
            we set all other user memberships/players is-current flag to false and set found player is-current to true
            This is needed to help the back end identify the player current game when the player is playing a game
            */

      const adventuresService = new ItemsService('adventures', {
        schema: req.schema,
        accountability: req.accountability
      })

      const teamsService = new ItemsService('teams', {
        schema: req.schema,
        accountability: req.accountability
      })

      const playersService = new ItemsService('players', {
        schema: req.schema,
        accountability: req.accountability
      })

      const responsesService = new ItemsService('responses', {
        schema: req.schema,
        accountability: req.accountability
      })

      try {
        const { teamId } = req.body
        const userId = req.accountability.user

        if (!teamId) {
          return res
            .status(400)
            .json({ success: false, error: 'teamId parameter is required.' })
        }
        const team = await teamsService.readOne(teamId)
        if (team == null) {
          return res
            .status(400)
            .json({ success: false, error: 'teamId is not found.' })
        }
        let existingPlayer = await playersService.readByQuery({
          filter: { user: { _eq: userId }, team: { _eq: teamId } },
          // filter: { user: { _eq: userId }, team: { _eq: teamId }, is_current: { _eq: true } },
          limit: -1
        })
        if (existingPlayer.length == 0) {
          return res
            .status(400)
            .json({
              success: false,
              error: 'You do not seem to be a member of this team!'
            })
        }
        if (existingPlayer[0].is_current == false) {
          return res
            .status(400)
            .json({
              success: false,
              error: 'You do not seem to have started the game yet.'
            })
        }

        let adventure = await adventuresService.readOne(team.adventure, {
          fields: [
            '*',
            'teams.*',
            'stages.*',
            'stages.qrcode.*',
            'stages.qrcode.qrcode_id.*',
            'stages.trivia.*',
            'stages.trivia.triviaquiz_id.*',
            'stages.sudoku.*',
            'stages.sudoku.sudoku_id.*',
            'stages.fifteenup.*',
            'stages.fifteenup.fifteenup_id.*',
            'stages.wordsearch.*',
            'stages.wordsearch.wordsearch_id.*',
            'stages.connectdots.*',
            'stages.connectdots.connectdots_id.*',
            'stages.riddles.*',
            'stages.riddles.riddles_id.*',
            'stages.quickmath.*',
            'stages.quickmath.quickmath_id.*',
            'stages.mathquiz.*',
            'stages.mathquiz.mathquiz_id.*',
            'stages.challenges.*',
            'stages.challenges.challenges_id.*',
            'stages.treasurehunt.*',
            'stages.treasurehunt.treasurehunt_id.*',
            'stages.spotthedifference.*',
            'stages.spotthedifference.spotthedifference_id.*',
          ]
        })

        let players = await playersService.readByQuery({
          filter: {
            team: {
              adventure: {
                id: {
                  _eq: team.adventure
                }
              }
            }
          },
          fields: ['*', 'user.first_name', 'user.last_name'],
          limit: -1
        })

        adventure['players'] = players

        if (new Date(adventure.start_date_and_time) > new Date()) {
          return res
            .status(400)
            .json({
              success: false,
              error: `The game did not start yet! It starts at ${adventure.start_date_and_time}`
            })
        }

        if (new Date(adventure.finish_date_and_time) < new Date()) {
          return res
            .status(400)
            .json({
              success: false,
              error: `The game is not available anymore! It finished at ${adventure.finish_date_and_time}`
            })
        }

        let responses = await responsesService.readByQuery({
          filter: { team: { _eq: teamId } },
          limit: -1
        })

        let games = [
          'trivia',
          'quickmath',
          'mathquiz',
          'sudoku',
          'connectdots',
          'wordsearch',
          'fifteenup',
          'riddles',
          'challenges',
          'treasurehunt',
          'spotthedifference',
          'qrcode',
        ]
        for (let i = 0; i < games.length; i++) {
          let game = games[i]
          let total_questions = 0
          let total_answers = 0
          for (let j = 0; j < adventure.stages.length; j++) {
            let stage = adventure.stages[j]
            
            if (!stage['questions_count']) {
              stage['questions_count'] = 0
            } 
            
            if (!stage['answers_count']) {
              stage['answers_count'] = 0
            } 

            for (let k = 0; k < stage[game].length; k++) {
              stage['questions_count'] += 1
              let m2m = stage[game][k]
              if (
                responses.filter(
                  resp =>
                    resp.game_name === game &&
                    resp.question_id === m2m.id &&
                    resp.team === teamId
                ).length > 0
              ) {
                adventure.stages[j][game][k]['has_response'] = true
                stage['answers_count'] += 1
                if (game == 'qrcode') {
                  stage['qrcode_scanned'] = 'yes'
                }
                if (game == 'riddles') {
                  stage['riddle_solved'] = 'yes'
                }

              } else {
                adventure.stages[j][game][k]['has_response'] = false
                if (game == 'qrcode') {
                  stage['qrcode_scanned'] = 'no'
                }
                if (game == 'riddles') {
                  stage['riddle_solved'] = 'no'
                }

              }
            }
          }
        }
        for (let i = 0; i < adventure.stages.length; i++) {
          let stage = adventure.stages[i]
          if (stage['questions_count'] == stage['answers_count']){
            stage['questions_answered'] = 'yes'
          } else {
            stage['questions_answered'] = 'no'
          }
          stage['stage_score'] = { total: 0 }

          for (let j = 0; j < games.length; j++) {
            let game = games[j]
            let game_score =
              responses
                .filter(
                  resp =>
                    resp.stage === stage['id'] &&
                    resp.game_name === game &&
                    resp.team === teamId
                )
                .reduce((sum, resp) => sum + resp.points, 0) ?? 0

            stage['stage_score'][game] = game_score
            stage['stage_score']['total'] += game_score
          }
          // adventure.stages[j]['stage_score'] = adventure.stages[j].reduce((sum, stage) => sum + stage.game_score, 0);
        }

        res.status(200).json({
          adventure: adventure,
          player: existingPlayer[0]
        })
      } catch (error) {
        res.status(500).json({
          success: false,
          error: error.message
        })
      }
    })

    router.post('/submit-response', async (req, res, next) => {
      const responsesService = new ItemsService('responses', {
        schema: req.schema,
        accountability: req.accountability
      })

      const playersService = new ItemsService('players', {
        schema: req.schema,
        accountability: req.accountability
      })

      const stagesService = new ItemsService('stages', {
        schema: req.schema,
        accountability: req.accountability
      })

      try {
        const {
          playerId,
          stageId,
          gameName,
          questionId,
          answer,
          isCorrect,
          elapsedTime,
          attachment,
          revealedHints
        } = req.body

        // Perform validation
        if (!playerId || !gameName || !questionId || !stageId) {
          return res.status(400).json({
            success: false,
            error: `Missing info: playerId=${playerId}, stageId=${stageId}, gameName=${gameName}, questionId=${questionId}`
          })
        }

        let player = await playersService.readByQuery({
          filter: { id: { _eq: playerId } },
          fields: ['*', 'team.*', 'user.*', 'team.adventure.*']
        })

        if (player.length === 0) {
          return res.status(400).json({
            success: false,
            error: `Player with ID ${playerId} not found.`
          })
        }

        // check player is submitting to his started game
        if (!player[0].is_current) {
          return res.status(400).json({
            success: false,
            error: `Player game is not started properly.`
          })
        }
        // check game is started and did not finish yet
        if (
          new Date(player[0].team.adventure.start_date_and_time) > new Date()
        ) {
          return res.status(400).json({
            success: false,
            error: `Game is not started yet.`
          })
        }
        if (
          new Date(player[0].team.adventure.finish_date_and_time) < new Date()
        ) {
          return res.status(400).json({
            success: false,
            error: `Game ended, no more answers will be accepted.`
          })
        }

        let frontendCorrection = true
        if (
          ![
            'trivia',
            'mathquiz',
            'quickmath',
            'sudoku',
            'fifteenup',
            'connectdots',
            'wordsearch',
            'riddles',
            'challenges',
            'treasurehunt',
            'spotthedifference',
            'qrcode'
          ].includes(gameName)
        ) {
          return res.status(400).json({
            success: false,
            error: `game ${gameName} not correcet.`
          })
        }

        if (
          [
            'trivia',
            'mathquiz',
            'quickmath',
            'sudoku',
            'fifteenup',
            'connectdots',
            'wordsearch',
            'riddles',
            'spotthedifference',
            'qrcode'
          ].includes(gameName)
        ) {
          if (isCorrect === null) {
            return res.status(400).json({
              success: false,
              error: `game ${gameName} requires parameter isCorrect to be set.`
            })
          }
        }

        if (['challenges', 'treasurehunt'].includes(gameName)) {
          if (attachment === null) {
            return res.status(400).json({
              success: false,
              error: `game ${gameName} requires an attachment to be uploaded.`
            })
          }
          frontendCorrection = false
        }
        if (['riddles'].includes(gameName)) {
          if (
            revealedHints === null ||
            revealedHints < 0 ||
            revealedHints > 3
          ) {
            return res.status(400).json({
              success: false,
              error: `gameName ${gameName} requires revealedHints to be set properly (between 0 and 3).`
            })
          }
        }
        let question_reference =
          gameName == 'trivia' ? 'triviaquiz_id' : gameName + '_id'

        let stage = await stagesService.readByQuery({
          filter: { id: { _eq: stageId } },
          fields: [
            '*',
            'adventure.*',
            `${gameName}.*`,
            `${gameName}.${question_reference}.*`
          ]
        })

        if (stage.length === 0) {
          return res.status(400).json({
            success: false,
            error: `stage with ID ${stageId} not found.`
          })
        }

        if (stage[0].adventure.id != player[0].team.adventure.id) {
          return res.status(400).json({
            success: false,
            error: `stage adventure ${stage[0].adventure.id} and player adventure ${player[0].team.adventure.id} are not compatible`
          })
        }
        let question = stage[0][gameName].filter(
          quest => quest.id === questionId
        )

        if (question.length === 0) {
          return res.status(400).json({
            success: false,
            stage: stage,
            error: `Question with type ${gameName} and ID ${questionId} not found in stage ${stageId}`
          })
        }

        // check if there is an existing response from any player in team
        const existingResponse = await responsesService.readByQuery({
          filter: {
            game_name: {
              _eq: gameName
            },
            question_id: {
              _eq: questionId
            },
            team: {
              _eq: player[0].team.id
            }
          },
          fields: ['*'],
          limit: -1
        })
        if (existingResponse.length > 0 && gameName != 'challenges') {
          return res.status(400).json({
            success: false,
            error: `This question is answered before by ${player[0].user.first_name}`
          })
        }

        // calculate gained points
        let gainedPoints = 0
        const difficultyPoints = {
          easy: 10,
          medium: 20,
          hard: 40,
          perfect: 60
        }
        if (isCorrect === true) {
          switch (gameName) {
            case 'trivia':
              gainedPoints = elapsedTime <= 5 ? 20 : 10
              break
            case 'quickmath':
              gainedPoints = elapsedTime <= 5 ? 20 : 10
              break
            case 'sudoku':
              gainedPoints = difficultyPoints[question.difficulty.toLowerCase()]
              break
            case 'wordsearch':
              gainedPoints = difficultyPoints[question.difficulty.toLowerCase()]
              break
            case 'connectdots':
              gainedPoints = difficultyPoints[question.difficulty.toLowerCase()]
              break
            case 'fifteenup':
              gainedPoints = difficultyPoints[question.difficulty.toLowerCase()]
              break
            case 'spotthedifference':
              gainedPoints = difficultyPoints[question.difficulty.toLowerCase()]
              break
            case 'riddles':
              gainedPoints = 50 - revealedHints ?? 0 * 10
              break
            default:
            // code block
          }
        }

        const newResponse = await responsesService.createOne({
          player: playerId,
          team: player[0].team.id,
          stage: stageId,
          game_name: gameName,
          question_id: questionId, // stage-repository m2m id
          repository_row_id: question[0][question_reference].id,
          answer: answer,
          attachment: attachment,
          is_corrected: frontendCorrection,
          is_correct: isCorrect,
          points: gainedPoints,
          response_time: new Date()
        })

        res.status(200).json({
          success: true,
          created_response: newResponse
        })
      } catch (error) {
        res.status(500).json({
          success: false,
          error: error.message
        })
      }
    })

    router.post('/get-progress', async (req, res, next) => {
      const adventuresService = new ItemsService('adventures', {
        schema: req.schema,
        accountability: req.accountability
      })

      const teamsService = new ItemsService('teams', {
        schema: req.schema,
        accountability: req.accountability
      })

      const playersService = new ItemsService('players', {
        schema: req.schema,
        accountability: req.accountability
      })

      const responsesService = new ItemsService('responses', {
        schema: req.schema,
        accountability: req.accountability
      })

      try {
        const startTime = new Date()
        const { teamId } = req.body
        const userId = req.accountability.user

        if (!teamId) {
          return res
            .status(400)
            .json({ success: false, error: 'Team code is required.' })
        }
        const team = await teamsService.readOne(teamId)

        const adventure = await adventuresService.readOne(team.adventure, {
          fields: [
            '*',
            'teams.*',
            'stages.*',
            'stages.trivia.*',
            'stages.trivia.triviaquiz_id.*',
            'stages.sudoku.*',
            'stages.sudoku.sudoku_id.*',
            'stages.fifteenup.*',
            'stages.fifteenup.fifteenup_id.*',
            'stages.wordsearch.*',
            'stages.wordsearch.wordsearch_id.*',
            'stages.connectdots.*',
            'stages.connectdots.connectdots_id.*',
            'stages.riddles.*',
            'stages.riddles.riddles_id.*',
            'stages.quickmath.*',
            'stages.quickmath.quickmath_id.*'
          ]
        })

        if (new Date(adventure.start_date_and_time) > new Date()) {
          return res
            .status(400)
            .json({ success: false, error: 'The game did not start yet!' })
        }
        if (new Date(adventure.finish_date_and_time) < new Date()) {
          return res
            .status(400)
            .json({
              success: false,
              error: 'The game is not available anymore!'
            })
        }

        let existingPlayer = await playersService.readByQuery({
          filter: { user: { _eq: userId }, team: { _eq: teamId } },
          limit: 1
        })

        if (existingPlayer.length == 0) {
          return res
            .status(400)
            .json({
              success: false,
              error: 'The player is not a member of the specified team!'
            })
        }

        let responses = await responsesService.readByQuery({
          filter: { team: { _eq: teamId } },
          limit: -1
        })

        let stages_progress = []
        for (let i = 0; i < adventure.stages.length; i++) {
          let stage = adventure.stages[i]
          let stage_progress = {
            stage: stage.id,

            trivia_count: stage.trivia.length,
            trivia_responses_count: responses.filter(
              response =>
                response.game_name === 'trivia' && response.stage === stage.id
            ).length,

            quickmath_count: stage.quickmath.length,
            quickmath_responses_count: responses.filter(
              response =>
                response.game_name === 'quickmath' &&
                response.stage === stage.id
            ).length,

            riddles_count: stage.riddles.length,
            riddles_responses_count: responses.filter(
              response =>
                response.game_name === 'riddles' && response.stage === stage.id
            ).length,

            challenges_count: stage.challenges.length,
            challenges_responses_count: responses.filter(
              response =>
                response.game_name === 'challenges' &&
                response.stage === stage.id
            ).length,

            treasurehunt_count: stage.treasurehunt.length,
            treasurehunt_responses_count: responses.filter(
              response =>
                response.game_name === 'treasurehunt' &&
                response.stage === stage.id
            ).length,

            activities_count:
              stage.sudoku.length +
              stage.wordsearch.length +
              stage.fifteenup.length +
              stage.connectdots.length,
            activities_responses_count: responses.filter(
              response =>
                ['sudoku', 'wordsearch', 'fifteenup', 'connectdots'].includes(
                  response.game_name
                ) && response.stage === stage.id
            ).length
          }
          stages_progress.push(stage_progress)
        }

        let adventure_progress = {
          trivia_count: 0,
          trivia_responses_count: 0,

          quickmath_count: 0,
          quickmath_responses_count: 0,

          riddles_count: 0,
          riddles_responses_count: 0,

          challenges_count: 0,
          challenges_responses_count: 0,

          treasurehunt_count: 0,
          treasurehunt_responses_count: 0,

          activities_count: 0,
          activities_responses_count: 0
        }
        for (const stage of stages_progress) {
          adventure_progress.trivia_count += stage.trivia_count
          adventure_progress.trivia_responses_count +=
            stage.trivia_responses_count

          adventure_progress.quickmath_count += stage.quickmath_count
          adventure_progress.quickmath_responses_count +=
            stage.quickmath_responses_count

          adventure_progress.riddles_count += stage.riddles_count
          adventure_progress.riddles_responses_count +=
            stage.riddles_responses_count

          adventure_progress.challenges_count += stage.challenges_count
          adventure_progress.challenges_responses_count +=
            stage.challenges_responses_count

          adventure_progress.treasurehunt_count += stage.treasurehunt_count
          adventure_progress.treasurehunt_responses_count +=
            stage.treasurehunt_responses_count

          adventure_progress.activities_count += stage.activities_count
          adventure_progress.activities_responses_count +=
            stage.activities_responses_count
        }
        const endTime = new Date()
        res.status(200).json({
          success: true,
          adventure_progress: adventure_progress,
          stages_progress: stages_progress
        })
      } catch (error) {
        res.status(500).json({ error: error.message, success: false })
      }
    })

    router.post('/my-games', async (req, res, next) => {
      /*
            A user submits a team id, 
            we validate the team exists and is for an ongoing (current time is between game start and end times)
            we validate a player exists for the user and team id
            we set all other user memberships/players is-current flag to false and set found player is-current to true
            This is needed to help the back end identify the player current game when the player is playing a game
            */

      const adventuresService = new ItemsService('adventures', {
        schema: req.schema,
        accountability: req.accountability
      })

      const teamsService = new ItemsService('teams', {
        schema: req.schema,
        accountability: req.accountability
      })

      const playersService = new ItemsService('players', {
        schema: req.schema,
        accountability: req.accountability
      })

      const responsesService = new ItemsService('responses', {
        schema: req.schema,
        accountability: req.accountability
      })

      try {
        const userId = req.accountability.user
        const myTeams = await playersService.readByQuery({
          fields: ['*', 'team.*'],
          filter: {
            user: {
              _eq: userId
            }
          },
          limit: -1
        })

        const adventuresArray = myTeams.map(object => object.team?.adventure)
        const teamsArray = myTeams.map(object => object.team?.id)

        let adventures = await adventuresService.readByQuery({
          filter: {
            id: {
              _in: adventuresArray
            }
          },
          fields: ['*', 'teams.*']
        })

        for (let i = 0; i < adventures.length; i++) {
          for (let j = 0; j < adventures[i].teams.length; j++) {
            if (teamsArray.includes(adventures[i].teams[j].id)) {
              adventures[i]['myteam'] = adventures[i].teams[j].id
            }
          }
        }
        res.status(200).json({
          adventures: adventures,
          teamsArray: teamsArray
        })
      } catch (error) {
        res.status(500).json({
          success: false,
          error: error.message
        })
      }
    })

    router.post('/load-scores', async (req, res, next) => {
      /*
            A user submits a team id, 
            we validate the team exists and is for an ongoing (current time is between game start and end times)
            we validate a player exists for the user and team id
            we set all other user memberships/players is-current flag to false and set found player is-current to true
            This is needed to help the back end identify the player current game when the player is playing a game
            */

      const adventuresService = new ItemsService('adventures', {
        schema: req.schema,
        accountability: req.accountability
      })

      const teamsService = new ItemsService('teams', {
        schema: req.schema,
        accountability: req.accountability
      })

      const playersService = new ItemsService('players', {
        schema: req.schema,
        accountability: req.accountability
      })

      const responsesService = new ItemsService('responses', {
        schema: req.schema,
        accountability: req.accountability
      })

      try {
        const { teamId } = req.body
        const userId = req.accountability.user

        if (!teamId) {
          return res
            .status(400)
            .json({ success: false, error: 'teamId parameter is required.' })
        }
        const team = await teamsService.readOne(teamId)
        if (team == null) {
          return res
            .status(400)
            .json({ success: false, error: 'teamId is not found.' })
        }
        let existingPlayer = await playersService.readByQuery({
          filter: { user: { _eq: userId }, team: { _eq: teamId } },
          // filter: { user: { _eq: userId }, team: { _eq: teamId }, is_current: { _eq: true } },
          limit: -1
        })
        if (existingPlayer.length == 0) {
          return res
            .status(400)
            .json({
              success: false,
              error: 'You do not seem to be a member of this team!'
            })
        }

        let adventure = await adventuresService.readOne(team.adventure, {
          fields: ['*']
        })

        if (new Date(adventure.start_date_and_time) > new Date()) {
          return res
            .status(400)
            .json({
              success: false,
              error: `The game did not start yet! It starts at ${adventure.start_date_and_time}`
            })
        }

        if (new Date(adventure.finish_date_and_time) < new Date()) {
          return res
            .status(400)
            .json({
              success: false,
              error: `The game is not available anymore! It finished at ${adventure.finish_date_and_time}`
            })
        }

        let responses = await responsesService.readByQuery({
          fields: [
            '*',
            'team.*',
            'player.*',
            'stage.title',
            'stage.location',
            'player.user.first_name',
            'player.user.last_name'
          ],
          filter: {
            team: {
              adventure: {
                _eq: team.adventure
              }
            }
          },
          limit: -1
        })

        let stages_scores = responses.reduce((acc, resp) => {
          let stage = resp.stage.location;
          let team = resp.team.id
          if (!acc[stage]) {
            acc[stage] = {}
          }
          if (!acc[stage][team]) {
            acc[stage][team] = {
              participants: [],
              score: 0
            }
          }
          acc[stage][team]['participants'].push(resp.player.user)
          acc[stage][team]['score'] += resp.points
          return acc
        }, {})

        let team_scores = responses
          .filter(resp => resp.team.id === teamId)
          .reduce((acc, resp) => {
            let stage = resp.stage.location
            let game = resp.game_name
            if (!acc[stage]) {
              acc[stage] = {}
            }
            if (!acc[stage][game]) {
              acc[stage][game] = {
                participants: [],
                score: 0
              }
            }
            acc[stage][game]['participants'].push(resp.player.user)
            acc[stage][game]['score'] += resp.points
            return acc
        }, {})

        res.status(200).json({
          team_scores: team_scores,
          stages_scores: stages_scores
        })
      } catch (error) {
        res.status(500).json({
          success: false,
          error: error.message
        })
      }
    })

    router.post('/load-responses', async (req, res, next) => {

      const adventuresService = new ItemsService('adventures', {
        schema: req.schema,
        accountability: req.accountability
      })

      const teamsService = new ItemsService('teams', {
        schema: req.schema,
        accountability: req.accountability
      })

      const playersService = new ItemsService('players', {
        schema: req.schema,
        accountability: req.accountability
      })

      const responsesService = new ItemsService('responses', {
        schema: req.schema,
        accountability: req.accountability
      })

      try {
        const { teamId, stageId, gameName} = req.body

        const userId = req.accountability.user

        if (!teamId) {
          return res
            .status(400)
            .json({ success: false, error: 'teamId parameter is required.' })
        }
        const team = await teamsService.readOne(teamId)
        if (team == null) {
          return res
            .status(400)
            .json({ success: false, error: 'teamId is not found.' })
        }
        let existingPlayer = await playersService.readByQuery({
          filter: { user: { _eq: userId }, team: { _eq: teamId } },
          // filter: { user: { _eq: userId }, team: { _eq: teamId }, is_current: { _eq: true } },
          limit: -1
        })
        if (existingPlayer.length == 0) {
          return res
            .status(400)
            .json({
              success: false,
              error: 'You do not seem to be a member of this team!'
            })
        }
        let filter = {
          team: {
            adventure: {
              _eq: team.adventure
            }
          }
        }
        if (stageId){
          filter['stage'] = {
            id: {
              _eq: stageId
            }
          }
        }
        if (gameName){
          filter['game_name'] = {
              _eq: gameName
          }
        }
        let responses = await responsesService.readByQuery({
          fields: ['*',],
          filter: filter,
          limit: -1
        })

        res.status(200).json({
          responses: responses,
        })
      } catch (error) {
        res.status(500).json({
          success: false,
          error: error.message
        })
      }
    })

    router.get('/', (req, res) => res.send('Playing endpoints!'))
  }
}
