import { InvalidPayloadError, UnprocessableContentError } from '@directus/errors';
import { getMatch } from 'ip-matching';
import { checkIncreasedUserLimits } from '../telemetry/utils/check-increased-user-limits.js';
import { getRoleCountsByUsers } from '../telemetry/utils/get-role-counts-by-users.js';
import {} from '../telemetry/utils/get-user-count.js';
import { getUserCountsByRoles } from '../telemetry/utils/get-user-counts-by-roles.js';
import { transaction } from '../utils/transaction.js';
import { ItemsService } from './items.js';
import { PermissionsService } from './permissions/index.js';
import { PresetsService } from './presets.js';
import { UsersService } from './users.js';
import { shouldClearCache } from '../utils/should-clear-cache.js';
import { omit } from 'lodash-es';
export class RolesService extends ItemsService {
    constructor(options) {
        super('directus_roles', options);
    }
    async checkForOtherAdminRoles(excludeKeys) {
        // Make sure there's at least one admin role left after this deletion is done
        const otherAdminRoles = await this.knex
            .count('*', { as: 'count' })
            .from('directus_roles')
            .whereNotIn('id', excludeKeys)
            .andWhere({ admin_access: true })
            .first();
        const otherAdminRolesCount = Number(otherAdminRoles?.count ?? 0);
        if (otherAdminRolesCount === 0) {
            throw new UnprocessableContentError({ reason: `You can't delete the last admin role` });
        }
    }
    async checkForOtherAdminUsers(key, users) {
        const role = await this.knex.select('admin_access').from('directus_roles').where('id', '=', key).first();
        // No-op if role doesn't exist
        if (!role)
            return;
        const usersBefore = (await this.knex.select('id').from('directus_users').where('role', '=', key)).map((user) => user.id);
        const usersAdded = [];
        const usersUpdated = [];
        const usersCreated = [];
        const usersRemoved = [];
        if (Array.isArray(users)) {
            const usersKept = [];
            for (const user of users) {
                if (typeof user === 'string') {
                    if (usersBefore.includes(user)) {
                        usersKept.push(user);
                    }
                    else {
                        usersAdded.push({ id: user });
                    }
                }
                else if (user.id) {
                    if (usersBefore.includes(user.id)) {
                        usersKept.push(user.id);
                        usersUpdated.push(user);
                    }
                    else {
                        usersAdded.push(user);
                    }
                }
                else {
                    usersCreated.push(user);
                }
            }
            usersRemoved.push(...usersBefore.filter((user) => !usersKept.includes(user)));
        }
        else {
            for (const user of users.update) {
                if (usersBefore.includes(user['id'])) {
                    usersUpdated.push(user);
                }
                else {
                    usersAdded.push(user);
                }
            }
            usersCreated.push(...users.create);
            usersRemoved.push(...users.delete);
        }
        if (role.admin_access === false || role.admin_access === 0) {
            // Admin users might have moved in from other role, thus becoming non-admin
            if (usersAdded.length > 0) {
                const otherAdminUsers = await this.knex
                    .count('*', { as: 'count' })
                    .from('directus_users')
                    .leftJoin('directus_roles', 'directus_users.role', 'directus_roles.id')
                    .whereNotIn('directus_users.id', usersAdded.map((user) => user.id))
                    .andWhere({ 'directus_roles.admin_access': true, status: 'active' })
                    .first();
                const otherAdminUsersCount = Number(otherAdminUsers?.count ?? 0);
                if (otherAdminUsersCount === 0) {
                    throw new UnprocessableContentError({ reason: `You can't remove the last admin user from the admin role` });
                }
            }
            return;
        }
        // Only added or created new users
        if (usersUpdated.length === 0 && usersRemoved.length === 0)
            return;
        // Active admin user(s) about to be created
        if (usersCreated.some((user) => !('status' in user) || user.status === 'active'))
            return;
        const usersDeactivated = [...usersAdded, ...usersUpdated]
            .filter((user) => 'status' in user && user.status !== 'active')
            .map((user) => user.id);
        const usersAddedNonDeactivated = usersAdded
            .filter((user) => !usersDeactivated.includes(user.id))
            .map((user) => user.id);
        // Active user(s) about to become admin
        if (usersAddedNonDeactivated.length > 0) {
            const userCount = await this.knex
                .count('*', { as: 'count' })
                .from('directus_users')
                .whereIn('id', usersAddedNonDeactivated)
                .andWhere({ status: 'active' })
                .first();
            if (Number(userCount?.count ?? 0) > 0) {
                return;
            }
        }
        const otherAdminUsers = await this.knex
            .count('*', { as: 'count' })
            .from('directus_users')
            .leftJoin('directus_roles', 'directus_users.role', 'directus_roles.id')
            .whereNotIn('directus_users.id', [...usersDeactivated, ...usersRemoved])
            .andWhere({ 'directus_roles.admin_access': true, status: 'active' })
            .first();
        const otherAdminUsersCount = Number(otherAdminUsers?.count ?? 0);
        if (otherAdminUsersCount === 0) {
            throw new UnprocessableContentError({ reason: `You can't remove the last admin user from the admin role` });
        }
        return;
    }
    isIpAccessValid(value) {
        if (value === undefined)
            return false;
        if (value === null)
            return true;
        if (Array.isArray(value) && value.length === 0)
            return true;
        for (const ip of value) {
            if (typeof ip !== 'string' || ip.includes('*'))
                return false;
            try {
                const match = getMatch(ip);
                if (match.type == 'IPMask')
                    return false;
            }
            catch {
                return false;
            }
        }
        return true;
    }
    assertValidIpAccess(partialItem) {
        if ('ip_access' in partialItem && !this.isIpAccessValid(partialItem['ip_access'])) {
            throw new InvalidPayloadError({
                reason: 'IP Access contains an incorrect value. Valid values are: IP addresses, IP ranges and CIDR blocks',
            });
        }
    }
    getRoleAccessType(data) {
        if ('admin_access' in data && data['admin_access'] === true) {
            return 'admin';
        }
        else if (('app_access' in data && data['app_access'] === true) || 'app_access' in data === false) {
            return 'app';
        }
        else {
            return 'api';
        }
    }
    async createOne(data, opts) {
        this.assertValidIpAccess(data);
        const increasedCounts = {
            admin: 0,
            app: 0,
            api: 0,
        };
        const existingIds = [];
        if ('users' in data) {
            const type = this.getRoleAccessType(data);
            increasedCounts[type] += data['users'].length;
            for (const user of data['users']) {
                if (typeof user === 'string') {
                    existingIds.push(user);
                }
                else if (typeof user === 'object' && 'id' in user) {
                    existingIds.push(user['id']);
                }
            }
        }
        await checkIncreasedUserLimits(this.knex, increasedCounts, existingIds);
        return super.createOne(data, opts);
    }
    async createMany(data, opts) {
        const increasedCounts = {
            admin: 0,
            app: 0,
            api: 0,
        };
        const existingIds = [];
        for (const partialItem of data) {
            this.assertValidIpAccess(partialItem);
            if ('users' in partialItem) {
                const type = this.getRoleAccessType(partialItem);
                increasedCounts[type] += partialItem['users'].length;
                for (const user of partialItem['users']) {
                    if (typeof user === 'string') {
                        existingIds.push(user);
                    }
                    else if (typeof user === 'object' && 'id' in user) {
                        existingIds.push(user['id']);
                    }
                }
            }
        }
        await checkIncreasedUserLimits(this.knex, increasedCounts, existingIds);
        return super.createMany(data, opts);
    }
    async updateOne(key, data, opts) {
        this.assertValidIpAccess(data);
        try {
            const increasedCounts = {
                admin: 0,
                app: 0,
                api: 0,
            };
            let increasedUsers = 0;
            const existingIds = [];
            let existingRole = await this.knex
                .count('directus_users.id', { as: 'count' })
                .select('directus_roles.admin_access', 'directus_roles.app_access')
                .from('directus_users')
                .where('directus_roles.id', '=', key)
                .andWhere('directus_users.status', '=', 'active')
                .leftJoin('directus_roles', 'directus_users.role', '=', 'directus_roles.id')
                .groupBy('directus_roles.admin_access', 'directus_roles.app_access')
                .first();
            if (!existingRole) {
                try {
                    const role = (await this.knex
                        .select('admin_access', 'app_access')
                        .from('directus_roles')
                        .where('id', '=', key)
                        .first()) ?? { admin_access: null, app_access: null };
                    existingRole = { count: 0, ...role };
                }
                catch {
                    existingRole = { count: 0, admin_access: null, app_access: null };
                }
            }
            if ('users' in data) {
                await this.checkForOtherAdminUsers(key, data['users']);
                const users = data['users'];
                if (Array.isArray(users)) {
                    increasedUsers = users.length - Number(existingRole.count);
                    for (const user of users) {
                        if (typeof user === 'string') {
                            existingIds.push(user);
                        }
                        else if (typeof user === 'object' && 'id' in user) {
                            existingIds.push(user['id']);
                        }
                    }
                }
                else {
                    increasedUsers += users.create.length;
                    increasedUsers -= users.delete.length;
                    const userIds = [];
                    for (const user of users.update) {
                        if ('status' in user) {
                            // account for users being activated and deactivated
                            if (user['status'] === 'active') {
                                increasedUsers++;
                            }
                            else {
                                increasedUsers--;
                            }
                        }
                        userIds.push(user.id);
                    }
                    try {
                        const existingCounts = await getRoleCountsByUsers(this.knex, userIds);
                        if (existingRole.admin_access) {
                            increasedUsers += existingCounts.app + existingCounts.api;
                        }
                        else if (existingRole.app_access) {
                            increasedUsers += existingCounts.admin + existingCounts.api;
                        }
                        else {
                            increasedUsers += existingCounts.admin + existingCounts.app;
                        }
                    }
                    catch {
                        // ignore failed user call
                    }
                }
            }
            let isAccessChanged = false;
            let accessType = 'api';
            if ('app_access' in data) {
                if (data['app_access'] === true) {
                    accessType = 'app';
                    if (!existingRole.app_access)
                        isAccessChanged = true;
                }
                else if (existingRole.app_access) {
                    isAccessChanged = true;
                }
            }
            else if (existingRole.app_access) {
                accessType = 'app';
            }
            if ('admin_access' in data) {
                if (data['admin_access'] === true) {
                    accessType = 'admin';
                    if (!existingRole.admin_access)
                        isAccessChanged = true;
                }
                else if (existingRole.admin_access) {
                    isAccessChanged = true;
                }
            }
            else if (existingRole.admin_access) {
                accessType = 'admin';
            }
            if (isAccessChanged) {
                increasedCounts[accessType] += Number(existingRole.count);
            }
            increasedCounts[accessType] += increasedUsers;
            await checkIncreasedUserLimits(this.knex, increasedCounts, existingIds);
        }
        catch (err) {
            (opts || (opts = {})).preMutationError = err;
        }
        return super.updateOne(key, data, opts);
    }
    async updateBatch(data, opts = {}) {
        for (const partialItem of data) {
            this.assertValidIpAccess(partialItem);
        }
        const primaryKeyField = this.schema.collections[this.collection].primary;
        if (!opts.mutationTracker) {
            opts.mutationTracker = this.createMutationTracker();
        }
        const keys = [];
        try {
            await transaction(this.knex, async (trx) => {
                const service = new RolesService({
                    accountability: this.accountability,
                    knex: trx,
                    schema: this.schema,
                });
                for (const item of data) {
                    const combinedOpts = Object.assign({ autoPurgeCache: false }, opts);
                    keys.push(await service.updateOne(item[primaryKeyField], omit(item, primaryKeyField), combinedOpts));
                }
            });
        }
        finally {
            if (shouldClearCache(this.cache, opts, this.collection)) {
                await this.cache.clear();
            }
        }
        return keys;
    }
    async updateMany(keys, data, opts) {
        this.assertValidIpAccess(data);
        try {
            if ('admin_access' in data && data['admin_access'] === false) {
                await this.checkForOtherAdminRoles(keys);
            }
            if ('admin_access' in data || 'app_access' in data) {
                const existingCounts = await getUserCountsByRoles(this.knex, keys);
                const increasedCounts = {
                    admin: 0,
                    app: 0,
                    api: 0,
                };
                const type = this.getRoleAccessType(data);
                for (const [existingType, existingCount] of Object.entries(existingCounts)) {
                    if (existingType === type)
                        continue;
                    increasedCounts[type] += existingCount;
                }
                await checkIncreasedUserLimits(this.knex, increasedCounts);
            }
        }
        catch (err) {
            (opts || (opts = {})).preMutationError = err;
        }
        return super.updateMany(keys, data, opts);
    }
    async updateByQuery(query, data, opts) {
        this.assertValidIpAccess(data);
        return super.updateByQuery(query, data, opts);
    }
    async deleteOne(key) {
        await this.deleteMany([key]);
        return key;
    }
    async deleteMany(keys) {
        const opts = {};
        try {
            await this.checkForOtherAdminRoles(keys);
        }
        catch (err) {
            opts.preMutationError = err;
        }
        await transaction(this.knex, async (trx) => {
            const itemsService = new ItemsService('directus_roles', {
                knex: trx,
                accountability: this.accountability,
                schema: this.schema,
            });
            const permissionsService = new PermissionsService({
                knex: trx,
                accountability: this.accountability,
                schema: this.schema,
            });
            const presetsService = new PresetsService({
                knex: trx,
                accountability: this.accountability,
                schema: this.schema,
            });
            const usersService = new UsersService({
                knex: trx,
                accountability: this.accountability,
                schema: this.schema,
            });
            // Delete permissions/presets for this role, suspend all remaining users in role
            await permissionsService.deleteByQuery({
                filter: { role: { _in: keys } },
            }, { ...opts, bypassLimits: true });
            await presetsService.deleteByQuery({
                filter: { role: { _in: keys } },
            }, { ...opts, bypassLimits: true });
            await usersService.updateByQuery({
                filter: { role: { _in: keys } },
            }, {
                status: 'suspended',
                role: null,
            }, { ...opts, bypassLimits: true });
            await itemsService.deleteMany(keys, opts);
        });
        return keys;
    }
    deleteByQuery(query, opts) {
        return super.deleteByQuery(query, opts);
    }
}
